<?php

namespace MPHB\Utils;

class ColorUtils {
	
	/**
	 * Hex darker/lighter/contrast functions for colours.
	 *
	 * @param mixed $color
	 * @return string
	 */
	public static function rgbFromHex( $color ){
		$color	 = str_replace( '#', '', $color );
		// Convert shorthand colors to full format, e.g. "FFF" -> "FFFFFF"
		$color	 = preg_replace( '~^(.)(.)(.)$~', '$1$1$2$2$3$3', $color );

		$rgb		 = array();
		$rgb['R']	 = hexdec( $color{0} . $color{1} );
		$rgb['G']	 = hexdec( $color{2} . $color{3} );
		$rgb['B']	 = hexdec( $color{4} . $color{5} );

		return $rgb;
	}

	/**
	 * Hex darker/lighter/contrast functions for colours.
	 *
	 * @param mixed $color
	 * @param int $factor (default: 30)
	 * @return string
	 */
	public static function hexDarker( $color, $factor = 30 ){
		$base	 = self::rgbFromHex( $color );
		$color	 = '#';

		foreach ( $base as $k => $v ) {
			$amount		 = $v / 100;
			$amount		 = round( $amount * $factor );
			$new_decimal = $v - $amount;

			$new_hex_component = dechex( $new_decimal );
			if ( strlen( $new_hex_component ) < 2 ) {
				$new_hex_component = "0" . $new_hex_component;
			}
			$color .= $new_hex_component;
		}

		return $color;
	}

	/**
	 * Hex darker/lighter/contrast functions for colours.
	 *
	 * @param mixed $color
	 * @param int $factor (default: 30)
	 * @return string
	 */
	public static function hexLighter( $color, $factor = 30 ){
		$base	 = self::rgbFromHex( $color );
		$color	 = '#';

		foreach ( $base as $k => $v ) {
			$amount		 = 255 - $v;
			$amount		 = $amount / 100;
			$amount		 = round( $amount * $factor );
			$new_decimal = $v + $amount;

			$new_hex_component = dechex( $new_decimal );
			if ( strlen( $new_hex_component ) < 2 ) {
				$new_hex_component = "0" . $new_hex_component;
			}
			$color .= $new_hex_component;
		}

		return $color;
	}

	/**
	 * Detect if we should use a light or dark colour on a background colour.
	 *
	 * @param mixed $color
	 * @param string $dark (default: '#000000')
	 * @param string $light (default: '#FFFFFF')
	 * @return string
	 */
	public static function lightOrDark( $color, $dark = '#000000', $light = '#FFFFFF' ){

		$hex = str_replace( '#', '', $color );

		$c_r = hexdec( substr( $hex, 0, 2 ) );
		$c_g = hexdec( substr( $hex, 2, 2 ) );
		$c_b = hexdec( substr( $hex, 4, 2 ) );

		$brightness = ( ( $c_r * 299 ) + ( $c_g * 587 ) + ( $c_b * 114 ) ) / 1000;

		return $brightness > 155 ? $dark : $light;
	}

	/**
	 * Format string as hex.
	 *
	 * @param string $hex
	 * @return string
	 */
	public static function formatHex( $hex ){

		$hex = trim( str_replace( '#', '', $hex ) );

		if ( strlen( $hex ) == 3 ) {
			$hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
		}

		return $hex ? '#' . $hex : null;
	}


}
